%% TESI Ialongo Leonardo
% A.A. 2023/2024

close all
clear
clc

%% Analisi statistica

Analisi_statistica = readtable("Analisi statistica supersonici.xlsx",Range="B1:X13",VariableNamingRule="preserve");

%% Parametri di progetto

Passeggeri = 80; %[50-80]
Payloadp = Passeggeri*120; %kg
Machp = 1.5;
Machsub = 0.85; %crociera subsonica
Rangep = 6000; %km [6000-7000]
Dp_max = 1.7; %psf
Hp = 18; %km [15-18]

%% Performance Fit

MTOM_fit = 95.9970*Payloadp^0.7982;                  %Case study MTOM
OEM_fit = 7.6546*MTOM_fit^0.7632;                    %Case study OEM
Fuel_fit = 0.0219*MTOM_fit^1.2544;                   %Case study Fuel
Thrust_fit = 180.5573*Machp^1.8295;                  %Case study Thrust
SFC_fit = 1.5948e-09*Thrust_fit^1.3130 + 3.2036e-05; %Case study SFC

%% Fuselage Fit

F_lenght = 0.0144*Payloadp^0.8076 + 34.1002; %Case study Fuselage lenght
F_width = 0.1306*F_lenght^0.7734;            %Case study Fuselage width
F_height = 0.1258*F_lenght^0.7883;           %Case study Fuselage height
A_height = 0.2316*F_lenght^0.9551;           %Case study Aircraft height

%% Wing Fit

W_area = 2.0398e-06*MTOM_fit^1.5676 + 84.8677; %Case study Wing area
W_span = 1.1753*W_area^0.5254;                 %Case study Wing span
W_rootC = 1.6955*W_area^0.4952;                %Case study Wing root chord
W_tipC = 0.1017*W_area^0.4951;                 %Case study Wing tip chord
W_sweepA = 52.9654*W_area^0.0328;              %Case study Wing mean sweep angle
W_AR = W_span^2/W_area;                        %Wing aspect ratio
W_MAC = 2/3 * (W_rootC^2+W_rootC*W_tipC+W_tipC^2)/(W_rootC+W_tipC); %Wing mean aerodinamic chord
W_lambda = W_tipC/W_rootC;                     %Wing taper ratio

%% Tail Fit

T_area = 0.0119*MTOM_fit^0.7296;   %Case study Tail area
T_span = 0.4598*T_area^0.5989;     %Case study Tail span
T_rootC = 2.7587*T_area^0.3528;    %Case study Tail root chord
T_tipC = 4.8780*T_area^-0.1135;    %Case study Tail tip chord
T_sweepA = 72.0803*T_area^-0.0545; %Case study Tail mean sweep angle
T_AR = T_span^2/T_area;            %Tail aspect ratio
T_MAC = 2/3 * (T_rootC^2+T_rootC*T_tipC+T_tipC^2)/(T_rootC+T_tipC); %Tail mean aerodinamic chord
T_lambda = T_tipC/T_rootC;         %Tail taper ratio

%% Vista in pianta

nose = 0.15*F_lenght;
tailcone = 0.8*F_lenght;
wingLE = 0.25*F_lenght;
tailLE = 0.7*F_lenght;

xfusoliera = [0 nose tailcone F_lenght]';
yfusoliera = [0 F_width/2 F_width/2 0]';

xala = [wingLE wingLE+W_rootC-W_tipC wingLE+W_rootC wingLE+W_rootC]';
yala = [0 W_span/2 W_span/2 0]';

xcoda = [tailLE tailLE+0.1*T_rootC tailLE+T_rootC-T_tipC tailLE+T_rootC]';
ycoda = [0 0.2 0.125 0]';
xc = [xcoda(1):0.1:xcoda(end)];
yc = interp1(xcoda,ycoda,xc,'spline');

mu = asin(1/Machp);
cone = mu*linspace(0,40);

figure(1)
plot(xfusoliera,yfusoliera,'k','LineWidth',1.5)
hold on
grid on
plot(xfusoliera,-yfusoliera,'k','LineWidth',1.5)
plot(xala,yala,'k','LineWidth',1.5)
plot(xala,-yala,'k','LineWidth',1.5)
plot(xc,yc,'k','LineWidth',1.5)
plot(xc,-yc,'k','LineWidth',1.5)
plot(cone,linspace(0,20),'r','LineWidth',1.5)
plot(cone,-linspace(0,20),'r','LineWidth',1.5)
xlim([-2 60])
title('Vista in pianta')


%% Profilo di missione

Crew = 4;         %2 piloti + 2 assistenti
Mcrew = Crew*120; %Massa totale della crew

T = 216.65; %Temperatura alla quota di crociera [K] (costante sopra gli 11km)
V(1) = Machsub*sqrt(1.4*287*T); %[m/s]
V(2) = Machp*sqrt(1.4*287*T); %[m/s]

WetAreaR = 3.1; %Wetted area ratio da tabella Raymer (Avro Vulcan)
K_ld = 15.5; %15.5 Civil jets; 14 Military jets

%Cruise
E_c = 0.866*K_ld*sqrt(W_AR/WetAreaR); %0.866*(L/D)max: valori tipici 7/7.5
%Loiter
E_l = K_ld*sqrt(W_AR/WetAreaR); %(L/D)max

%Endurance/Loiter time
End = 20*60; %sec (20 min)

%Range sola crociera subsonica
Range_cruisesub = 500*10^3; %[m]
%Range sola crociera supersonica
Range_cruisesup = (Rangep-Range_cruisesub/10^3-605)*10^3; %[m] 4200e3 Ale

K_all = 1.06; %Fattore di riserva del 6%

%Frazioni Wi/Wi-1 prese dal Raymer
Mf_to = 0.97;
% Mf_climb = 0.985;
Mf_climbsub = 0.96; %125km
Mf_climbsup = 0.95; %175km
Mf_descent = 0.931; %300km
Mf_land = 0.995;

Mf_cruisesub = exp(-Range_cruisesub*SFC_fit*9.81/(V(1)*E_c)); %Frazione fuel crociera subsonica (Breguet)
Mf_cruisesup = exp(-Range_cruisesup*SFC_fit*9.81/(V(2)*E_c)); %Frazione fuel crociera supersonica (Breguet)
Mf_loiter = exp(-End*SFC_fit*9.81/E_l);              %Frazione fuel loiter

Mf_tot = Mf_to*Mf_climbsub*Mf_climbsup*Mf_cruisesub*Mf_cruisesup*Mf_descent*Mf_loiter*Mf_land; %Massa fuel totale
Mf = (1-Mf_tot); %K_all*
% Mf_Ale = 0.97*0.999*0.957*0.924*0.947*0.668*0.931*0.985*0.982*0.995;

%% Parametrizzazione profilo di missione con il tempo

Hsub = 12; %quota crociera subsonica
to = 3;
sub_cl = 125;
sup_cl = 175;
sup_desc = 150;
sub_desc = 150;
land = 2;

Quota_fasi = [0 Hsub Hsub Hp Hp Hsub Hsub 0 0];
Range_fasi = [to sub_cl (Range_cruisesub/10^3)/2 sup_cl Range_cruisesup/10^3 sup_desc (Range_cruisesub/10^3)/2 sub_desc land];
V_fasi = [0 270/3.6 V(1) V(1) V(2) V(2) V(1) V(1) 250/3.6 0];

Range = zeros(9,1);
Quota = zeros(9,1);
t_fasi = ["Take off" "Subsonic climb" "Subsonic cruise" "Supersonic climb" "Supersonic cruise" "Descent 1" "Subsonic cruise" "Descent 2" "Landing"
    0 0 0 0 0 0 0 0 0]';
t_tot = 0;

disp('-------------------------------------------------------------------')
disp('FASI DI VOLO');
disp('-------------------------------------------------------------------')

for fase = 1:9
    
    Range(fase+1) = Range(fase) + Range_fasi(fase);
    Quota(fase+1) = Quota_fasi(fase);
    V_media = abs(V_fasi(fase+1)+V_fasi(fase))/2;
    t_fasi(fase,2) = (Range_fasi(fase)*10^3/V_media)/60; %[minuti]
    t_tot = t_tot + str2double(t_fasi(fase,2));
    
    fprintf('%s: %.1f minuti.\n',t_fasi(fase,1),t_fasi(fase,2));

end

fprintf('\nDurata complessiva del volo: %.1f minuti.\n\n',t_tot);

figure(2)
plot(Range,Quota,'lineWidth',1.5)
grid on
xlim([-300,6300])
ylim([-2,20])
xlabel('Range [km]')
ylabel('Quota [km]')

%% Metodo iterativo

MTOM_iter = @(x) (Payloadp+Mcrew)./(1-Mf-0.9*x.^(-0.06)); %Funzione MTOM da iterare: 0.97 -> 0.9

x = 140e3; %Valore iniziale
i = 1;
maxit = 15;
M0_calc = zeros(maxit,1);  %MTOM calcolati ogni iterazione
M0_guess = zeros(maxit,1); %MTOM di tentativo calcolato all'iterazione precedente

while (i <= maxit)
    
    M0_guess(i) = x;
    x = MTOM_iter(x);
    M0_calc(i) = x;
    i = i+1;
    
end

MTOM_conv = x;

%Grafico
M0 = linspace(100e3,250e3,1000);

figure(3)
loglog(M0,MTOM_iter(M0),'lineWidth',1.5)
hold on
loglog(M0,M0,'r','lineWidth',1.5)
grid on
xlabel('MTOM di tentativo [kg]')
ylabel('MTOM calcolato [kg]')
legend('Soluzione','Stima')

disp('-------------------------------------------------------------------')
disp('CONFRONTO MTOM');
disp('-------------------------------------------------------------------')
disp(' ');
disp(['MTOM da analisi statistica: ',num2str(MTOM_fit),' kg']);
disp(' ');
disp(['MTOM da iterazione: ',num2str(MTOM_conv),' kg']);

%% Aggiornamento dati

% %Performance
% OEM_new = 7.6546*MTOM_conv^0.7632;  %Case study OEM
% Fuel_new = 0.0219*MTOM_conv^1.2544; %Case study Fuel

%Wing
W_area = 2.0398e-06*MTOM_conv^1.5676 + 84.8677; %Case study Wing area
W_span = 1.1753*W_area^0.5254;                  %Case study Wing span
W_rootC = 1.6955*W_area^0.4952;                 %Case study Wing root chord
W_tipC = 0.1017*W_area^0.4951;                  %Case study Wing tip chord
W_sweepA = 52.9654*W_area^0.0328;               %Case study Wing mean sweep angle
W_AR = W_span^2/W_area;                         %Wing aspect ratio
W_MAC = 2/3 * (W_rootC^2+W_rootC*W_tipC+W_tipC^2)/(W_rootC+W_tipC); %Wing mean aerodinamic chord
W_lambda = W_tipC/W_rootC;                      %Wing taper ratio

%Tail
T_area = 0.0119*MTOM_conv^0.7296;  %Case study Tail area
T_span = 0.4598*T_area^0.5989;     %Case study Tail span
T_rootC = 2.7587*T_area^0.3528;    %Case study Tail root chord
T_tipC = 4.8780*T_area^-0.1135;    %Case study Tail tip chord
T_sweepA = 72.0803*T_area^-0.0545; %Case study Tail mean sweep angle
T_AR = T_span^2/T_area;            %Tail aspect ratio
T_MAC = 2/3 * (T_rootC^2+T_rootC*T_tipC+T_tipC^2)/(T_rootC+T_tipC); %Tail mean aerodinamic chord
T_lambda = T_tipC/T_rootC;         %Tail taper ratio

%% Calcolo Cl e Cd

%Condizioni di volo
mach = [0.3 0.3 0.6 Machsub 0.95 Machp 0.4 0.6 0.3];
h = [100 150 5000 12000 15000 Hp*1000 2000 5000 50];
alpha = [20 15 8 3 5 1 1 1 3]*pi/180;
man = [0.95 0.95 0.9 0.8 1 0.75 0.6 0.6 0.4];
n = length(mach);

% rho = [1.225 1.225 0.736 0.31 0.193 0.12 1.007 0.736 1.225];
% mu = [1.79 1.79 1.64 1.45 1.45 1.45 1.73 1.64 1.79]*10^-5;
% a = [341 341 320.53 295.07 295.07 295.07 332.53 320.53 341];
[T,a,P,rho,nu,mu] = atmosisa(h);
v = mach.*a;

% Geometry
eff = 0.95;                                                        % airfoil efficienty [-]
F_d = (F_width + F_height) / 2;                                    % fuselage average diameter [m]
% W_exp = W_area - (W_rootC * F_d);                                  % fuselage-wing exposed area [m2]
W_exp = W_area - 2*(W_rootC*F_d/2 - F_width^2/8*tan(W_sweepA*pi/180));  
F_area = pi * ((F_d / 2)^2);                                       % fuselage front area [m2]
F_wet = pi*F_d*F_lenght;                                           % fuselage wet area [m2]
Fw_wet = 2 * (W_area - (W_rootC * F_d));                           % fuselage-wing wet area [m2]
T_wet = 2 * T_area;                                                % tail wet area [m2] 
x_c = 0.5;                                                         % maximum thickness chord coordinate
tc_wing = 0.03;                                                    % airfoil thickness (wing) 
k_r = 0.17e-5;                                                     % roughness
tc_tail = 0.035;                                                   % airfoil thickness (vertical tail)
e_wd = 1.8;                                                        % empircal wave drag efficiency factor
n_eng = 4;                                                         % engine number [-]
l_nacelle = 9.5*0.8; % 80% del Concorde                            % nacelle lenght [m]
d_nacelle = 1.8*0.8; % 80% del Concorde                            % nacelle diameter [m]
swet_nacelle = 4*pi*l_nacelle*d_nacelle;                           % nacelle wet area [m2]

% LIFT

beta = sqrt(abs(1-mach.^2));

F = 1.07*(1+F_d/W_span)^2; %Fattore di portanza della fusoliera
Fcor = (W_exp/W_area)*F;
%(W_exp/W_area)*F dovrebbe essere poco meno di 1 altrimenti è come se la fusoliera producesse lift -> 0.98
%esce 1.0183
if Fcor>=1
    Fcor = 0.98;
end

for i=1:n
    if mach(i)<=0.8
        Cl_alpha(i) = (2*pi*W_AR)/(2+sqrt(4+(W_AR^2*beta(i)^2/eff^2)*(1+tan(W_sweepA*pi/180)^2/beta(i)^2))) * Fcor;
    elseif mach(i)>=1.2
        Cl_alpha(i) = 4/beta(i);
    else
        beta1 = sqrt(abs(1-0.8^2));
        beta2 = sqrt(abs(1-1.2^2));
        Cl_alpha1 = (2*pi*W_AR)/(2+sqrt(4+(W_AR^2*beta1^2/eff^2)*(1+tan(W_sweepA*pi/180)^2/beta1^2))) * Fcor;
        Cl_alpha2 = 4/beta2;
        Cl_alpha(i) = interp1([0.8, 1.2], [Cl_alpha1, Cl_alpha2], mach(i), 'linear');
    end
    Cl(i) = Cl_alpha(i) * alpha(i);
end
% Cl(6) = 2.4093 * alpha(6);

% DRAG

l_ref = [F_lenght, W_MAC, T_MAC, l_nacelle];
l = length(l_ref);


%cf
for j=1:n
    for i=1:l
        Rey = l_ref(i)*rho(j)*mach(j)*a(j)/mu(j);
        if mach(j)<=1
            Rey_cutoff  = 38.21 * (l_ref(i)/k_r)^1.053;
        elseif mach(j)>1
            Rey_cutoff  = 44.62 * (l_ref(i)/k_r)^1.053 * (mach(j)^1.16);
        end

        if Rey<=1.5e6 && Rey~=0
            cf(i,j) = 1.328/sqrt(Rey);
        elseif Rey>1.5e6
            cf(i,j) = 0.455/(log10(min(Rey_cutoff,Rey))^2.58 * (1+0.144*mach(j)^2)^0.65);
        else
            cf(i,j) = 0;
        end
    end
end


%ff

for i=1:n
    ff(1,i) = (1+(0.6/x_c)*tc_wing + 100*tc_wing^4) * (1.34*mach(i)^0.18*cos(W_sweepA*pi/180)^0.28); %wing
    ff(2,i) = 0.9 + 5/(F_lenght/F_d)^1.5 + (F_lenght/F_d)/400;                                       %fuselage
    ff(3,i) = (1+(0.6/x_c)*tc_tail + 100*tc_tail^4) * (1.34*mach(i)^0.18*cos(T_sweepA*pi/180)^0.28); %tail
    ff(4,i) = 1 + 0.35/(l_nacelle/d_nacelle);                                                        %nacelle
end

% %Cd_misc
% A_base = 0; %flap base area ???
% 
% %Subsonic
% Cd_misc(1) = ((0.139 + 0.419*(mach(1) - 0.161)^2) * A_base) / W_area;
% 
% %Supersonic
% Cd_misc(2) = ((0.064 + 0.042*(mach(2) - 3.840)^2) * A_base) / W_area;


%Cd_wave (only supersonic)

dq_sh = 9*pi/2 * (F_area/F_lenght)^2;
for i=1:n
    if mach(i)>1
        dq_wave = dq_sh * e_wd * (1 - 0.386 * (mach(i)-1.2)^0.57 * (1-(pi*(W_sweepA^0.77)/100)));
        cd_wave(i) = dq_wave/W_area;
    elseif mach(i)<=1
        cd_wave(i) = 0;
    end
end


%k

e = 4.61 * (1-0.045*W_AR^0.68) * ((cos(W_sweepA*pi/180))^0.15) - 3.1;
% e = 0.7;
for i=1:n
    if mach(i)<=0.8
        k(i) = 1/(pi*W_AR*e);
    elseif mach(i)>=1.2
        k(i) = (W_AR * (mach(i)^2 - 1) * cos(W_sweepA*pi/180)) / (4*W_AR * sqrt(mach(i)^2 - 1) - 2);
        % k(i) = 1.5*(W_AR * (mach(i)^2 - 1) * cos(W_sweepA*pi/180)) / (4*W_AR + sqrt(mach(i)^2 - 1) - 1);
    else
        k1 = 1/(pi*W_AR*e);
        k2 = (W_AR * (1.2^2 - 1) * cos(W_sweepA*pi/180)) / (4*W_AR * sqrt(1.2^2 - 1) - 2);
        k(i) = interp1([0.8, 1.2], [k1, k2], mach(i), 'linear');
    end
end


%Cd0

Q = [1, 1, 1.04, 1.5]; %interference factor
A_wet = [Fw_wet, F_wet, T_wet, swet_nacelle];

for i=1:n
    sum = 0;

    if mach(i)<=1
        for j=1:l
        sum = sum + cf(j,i)*ff(j)*Q(j)*A_wet(j);
        end
    elseif mach(i)>1
        sum = sum + cf(j,i)*A_wet(j);
    end

    if mach(i)<=1
        Cd0(i) = sum/W_area * (1+0.07);
    elseif mach(i)>1
        Cd0(i) = sum/W_area * (1+0.07) + cd_wave(i);
    end
end


%Cd

for i=1:n
        Cd(i) = Cd0(i) + k(i)*Cl(i)^2;
end


%% Analisi firme di pressione

data = readtable('Analisi firme pressione.xlsx', 'Sheet', 'Foglio1','VariableNamingRule','preserve');

X = data{:, {'Mach', 'Quota', 'Angolo d''attacco', 'Lunghezza', 'Apertura alare', 'Superficie alare'}};

% Matrice di correlazione delle variabili indipendenti
correlationMatrix = corrcoef(X);
disp(' ');
disp('Matrice di correlazione:');
disp(correlationMatrix);
disp(' ');
figure()
heatmap(correlationMatrix, 'ColorMap', parula, 'XDisplayLabels', {'Mach', 'Quota', 'Angolo d''attacco', 'Lunghezza', 'Apertura alare', 'Superficie alare'}, 'YDisplayLabels', {'Mach', 'Quota', 'Angolo d''attacco', 'Lunghezza', 'Apertura alare', 'Superficie alare'});

%% Divisione 2 categorie

%% CONVENZIONALE

% input e output
% 'Superficie alare' come variabile correlata
X = data{:, {'Mach', 'Quota', 'Angolo d''attacco', 'Superficie alare'}};
y1 = data.Dp1;  % Output Dp1
y2 = data.Dp2;  % Output Dp2

% Aggiunta intercetta
X = [ones(size(X, 1), 1), X];

% Regressione Lineare per Dp1
b1_linear = X \ y1;

% Previsioni per Dp1
y_pred1 = X * b1_linear;

difference1 = y1 - y_pred1;
SSE1 = 0;
for i = 1:length(difference1)
    SSE1 = SSE1 + difference1(i)^2;
end

mean_y1 = mean(y1);
SST1 = 0;
for i = 1:length(y1)
    SST1 = SST1 + (y1(i) - mean_y1)^2;
end

% Calcolo di R^2 per Dp1
R2_1 = 1 - (SSE1 / SST1);

% Risultati per Dp1
fprintf('Coefficiente di determinazione R^2 per Dp1: %.4f\n', R2_1);
fprintf('Equazione per Dp1: Dp1 = %.4f + %.4f*Mach + %.4f*Quota + %.4f*Angolo d''attacco + %.4f*Superficie alare\n', ...
    b1_linear(1), b1_linear(2), b1_linear(3), b1_linear(4), b1_linear(5));
disp(' ');

% Regressione Lineare per Dp2
b2_linear = X \ y2;

% Previsioni per Dp2
y_pred2 = X * b2_linear;

difference2 = y2 - y_pred2;
SSE2 = 0;
for i = 1:length(difference2)
    SSE2 = SSE2 + difference2(i)^2;
end

mean_y2 = mean(y2);
SST2 = 0;
for i = 1:length(y2)
    SST2 = SST2 + (y2(i) - mean_y2)^2;
end

% Calcolo di R^2 per Dp2
R2_2 = 1 - (SSE2 / SST2);  % Coefficiente di determinazione

% Risultati per Dp2
fprintf('Coefficiente di determinazione R^2 per Dp2: %.4f\n', R2_2);
fprintf('Equazione per Dp2: Dp2 = %.4f + %.4f*Mach + %.4f*Quota + %.4f*Angolo d''attacco + %.4f*Superficie alare\n', ...
    b2_linear(1), b2_linear(2), b2_linear(3), b2_linear(4), b2_linear(5));
disp(' ');


% %% LOW-BOOM
% 
% data = readtable('Analisi firme pressione.xlsx', 'Sheet', 'Foglio2');
% 
% % input e output
% % 'Superficie alare' come variabile correlata
% X = data{:, {'Mach', 'Quota', 'Angolo d''attacco', 'Superficie alare'}};
% y1 = data.Dp1;  % Output Dp1
% y2 = data.Dp2;  % Output Dp2
% 
% % Aggiunta intercetta
% X = [ones(size(X, 1), 1), X];
% 
% % Regressione Lineare per Dp1
% b1_linear = X \ y1;
% 
% % Previsioni per Dp1
% y_pred1 = X * b1_linear;
% 
% difference1 = y1 - y_pred1;
% SSE1 = 0;
% for i = 1:length(difference1)
%     SSE1 = SSE1 + difference1(i)^2;
% end
% 
% mean_y1 = mean(y1);
% SST1 = 0;
% for i = 1:length(y1)
%     SST1 = SST1 + (y1(i) - mean_y1)^2;
% end
% 
% % Calcolo di R^2 per Dp1
% R2_1 = 1 - (SSE1 / SST1);
% 
% % Risultati per Dp1
% fprintf('Coefficiente di determinazione R^2 per Dp1: %.4f\n', R2_1);
% fprintf('Equazione per Dp1: Dp1 = %.4f + %.4f*Mach + %.4f*Quota + %.4f*Angolo d''attacco + %.4f*Superficie alare\n', ...
%     b1_linear(1), b1_linear(2), b1_linear(3), b1_linear(4), b1_linear(5));
% disp(' ');
% 
% % Regressione Lineare per Dp2
% b2_linear = X \ y2;
% 
% % Previsioni per Dp2
% y_pred2 = X * b2_linear;
% 
% difference2 = y2 - y_pred2;
% SSE2 = 0;
% for i = 1:length(difference2)
%     SSE2 = SSE2 + difference2(i)^2;
% end
% 
% mean_y2 = mean(y2);
% SST2 = 0;
% for i = 1:length(y2)
%     SST2 = SST2 + (y2(i) - mean_y2)^2;
% end
% 
% % Calcolo di R^2 per Dp2
% R2_2 = 1 - (SSE2 / SST2);  % Coefficiente di determinazione
% 
% % Risultati per Dp2
% fprintf('Coefficiente di determinazione R^2 per Dp2: %.4f\n', R2_2);
% fprintf('Equazione per Dp2: Dp2 = %.4f + %.4f*Mach + %.4f*Quota + %.4f*Angolo d''attacco + %.4f*Superficie alare\n', ...
%     b2_linear(1), b2_linear(2), b2_linear(3), b2_linear(4), b2_linear(5));
% disp(' ');

%% Matching chart

%Requisiti
l_to = 2000;         % maximum take-off distance [m]
g_2 = 0.04;          % 2nd segment climb gradient [%] ref:2%
cr_sub = 10;         % subsonic cruise climb rate [m/s]
g_sub = cr_sub/v(3); % subsonic cruise climb gradient [%]
cr_sup = 1.5;        % supersonic cruise climb rate [m/s]
g_sup = cr_sup/v(5); % supersonic cruise climb gradient [%]
l_land = 3500;       % maximum landing field lenght [m]

sigma = rho./1.225;     % density ratio [-]
q = 0.5 .* rho .* v.^2; % dynamic pressure [Pa]

%Decollo
ws = 0:0.1:800;
tw_1 = (1 / (rho(1) * sigma(1) * l_to * Cl(1))) * ws;

figure()
plot(ws, tw_1,'DisplayName','Decollo','lineWidth',1.5)
hold on
legend
title('Matching Chart');
xlabel('W/S [kg/m^2]');
ylabel('T/W [-]');
xlim([0, 800]);
ylim([0, 5]);
grid on


%Secondo segmento
tw_2 = (n_eng / (n_eng - 1)) * (1 / (Cl(2) / Cd(2)) + g_2) * 1 / sigma(2);
plot([0, 800], [tw_2, tw_2], 'DisplayName', 'Secondo segmento','lineWidth',1.5)


%Salita subsonica
tw_3 = (q(3) * Cd(3) / (9.81 * man(3) * sigma(3))) * (1 ./ ws) + g_sub * (1 / (man(3) * sigma(3)));
plot(ws, tw_3,'DisplayName','Salita subsonica','lineWidth',1.5)


%Crociera subsonica
tw_4 = (q(4) * Cd(4)) ./ (9.81 * man(4) * sigma(4) .* ws);
plot(ws, tw_4,'r','DisplayName','Crociera subsonica','lineWidth',1.5)


%Salita supersonica
tw_5 = (q(5) * Cd(5) / (9.81 * man(5) * sigma(5))) * (1 ./ ws) + g_sup * (1 / (man(5) * sigma(5)));
plot(ws, tw_5,'DisplayName','Salita supersonica','lineWidth',1.5)


%Crociera supersonica
tw_6 = (q(6) * Cd(6)) ./ (9.81 * man(6) * sigma(6) .* ws);
plot(ws, tw_6,'DisplayName','Crociera supersonica','lineWidth',1.5);


%Virata istantanea
lf = 2.5; % load factor [-]
Cl_max = 1.18;

ws_7 = (q(7) * Cl_max) / (9.81 * lf * sigma(7));
plot([ws_7, ws_7], [0, 5], '--', 'DisplayName', 'Virata istantanea','lineWidth',1.5)


%Virata sostenuta
lf_s = 1.5;

tw_8 = ((q(8) * Cd0(8)) / (9.81 * sigma(8))) * (1 ./ ws) + ((sigma(8) * 9.81 * lf_s^2) / (q(8) * man(8) * pi * W_AR * e)) .* ws;
plot(ws, tw_8,'k', 'DisplayName', 'Virata sostenuta','lineWidth',1.5)


%Atterraggio
k_l = 0.149; % lofting parameter [kg/m^3]
Cl_max = 0.8;

ws_9 = k_l * sigma(9) * Cl_max * l_land;
plot([ws_9, ws_9], [0, 5], 'LineStyle', '--', 'DisplayName', 'Atterraggio','lineWidth',1.5)


%Boom sonico
Dp = 0.85;
ws_10 = (MTOM_conv * b1_linear(5)) / (Dp - (b1_linear(1) + b1_linear(2)*mach(6) + b1_linear(3)*h(6)/1000 + b1_linear(4)*alpha(6)*180/pi));
plot([ws_10, ws_10], [0, 5], 'LineStyle', '--', 'DisplayName', 'Boom sonico', 'LineWidth',1.5)

% tw_10 = (T_des*10^3 * b1_linear(5)) ./ (ws .* 9.81 .* (Dp - (b1_linear(1) + b1_linear(2)*mach(6) + b1_linear(3)*h(6)/1000 + b1_linear(4)*alpha(6)*180/pi)));
% plot(ws, tw_10, 'LineStyle', '-', 'DisplayName', 'Dp+', 'LineWidth',1.5)


%Emissioni acustiche LTO
tw_11 = 0.5850 - 4.2009e-4 .* ws;
plot(ws, tw_11, 'LineStyle', '-', 'DisplayName', 'Emissioni acustiche LTO','lineWidth',1.5)
hold off

%Calcolo spinta e nuova superficie alare
TW = 0.337688;
WS = 415.2;

T_des = TW*MTOM_conv*9.81/1000;
W_areades = MTOM_conv/WS;                          % Wing area desiderata [m^2]
T_despm = T_des/(4);                               % Spinta desiderata per motore [kN]


%% Zoom Matching chart

%Decollo
figure()
plot(ws, tw_1,'DisplayName','Decollo','lineWidth',1.5)
hold on
legend
title('Zoom Matching Chart');
xlabel('W/S [kg/m^2]');
ylabel('T/W [-]');
xlim([200, 500]);
ylim([0, 1]);
grid on


%Secondo segmento
plot([0, 800], [tw_2, tw_2], 'DisplayName', 'Secondo segmento','lineWidth',1.5)


%Salita subsonica
tw_3 = (q(3) * Cd(3) / (9.81 * man(3) * sigma(3))) * (1 ./ ws) + g_sub * (1 / (man(3) * sigma(3)));
plot(ws, tw_3,'DisplayName','Salita subsonica','lineWidth',1.5)


%Crociera subsonica
tw_4 = (q(4) * Cd(4)) ./ (9.81 * man(4) * sigma(4) .* ws);
plot(ws, tw_4,'r','DisplayName','Crociera subsonica','lineWidth',1.5)


%Salita supersonica
plot(ws, tw_5,'DisplayName','Salita supersonica','lineWidth',1.5)


%Crociera supersonica
tw_6 = (q(6) * Cd(6)) ./ (9.81 * man(6) * sigma(6) .* ws);
plot(ws, tw_6,'DisplayName','Crociera supersonica','lineWidth',1.5);


%Virata istantanea
plot([ws_7, ws_7], [0, 5], '--', 'DisplayName', 'Virata istantanea','lineWidth',1.5)


%Virata sostenuta
plot(ws, tw_8,'k', 'DisplayName', 'Virata sostenuta','lineWidth',1.5)


%Atterraggio
plot([ws_9, ws_9], [0, 5], 'LineStyle', '--', 'DisplayName', 'Atterraggio','lineWidth',1.5)


%Boom sonico
plot([ws_10, ws_10], [0, 5], 'LineStyle', '--', 'DisplayName', 'Boom sonico', 'LineWidth',1.5)


%Emissioni acustiche LTO
plot(ws, tw_11, 'LineStyle', '-', 'DisplayName', 'Emissioni acustiche LTO','lineWidth',1.5)
hold off

%% Dimensionamento finale

W_span = 1.1753*W_areades^0.5254;                  % Wing span
W_rootC = 1.6955*W_areades^0.4952;                 % Wing root chord
W_tipC = 0.1017*W_areades^0.4951;                  % Wing tip chord
W_sweepA = 52.9654*W_areades^0.0328;               % Wing mean sweep angle
W_AR = W_span^2/W_areades;                         % Wing aspect ratio
W_MAC = 2/3 * (W_rootC^2+W_rootC*W_tipC+W_tipC^2)/(W_rootC+W_tipC); % Wing mean aerodinamic chord
W_lambda = W_tipC/W_rootC;                         % Wing taper ratio

%% Calcolo dei picchi di pressione

%Convenzionale
Dp1c = b1_linear(1) + b1_linear(2)*mach(6) + b1_linear(3)*h(6)/1000 + b1_linear(4)*alpha(6)*180/pi + b1_linear(5)*W_areades;
Dp2c = b2_linear(1) + b2_linear(2)*mach(6) + b2_linear(3)*h(6)/1000 + b2_linear(4)*alpha(6)*180/pi + b2_linear(5)*W_areades;

%Low-boom
% Dp1lb = -0.4596*mach(6) + 0.0283*h(6)/1000 + 0.1134*alpha(6)*180/pi + 0.0052*W_areades;
% Dp2lb = -0.4000*mach(6) + -0.0157*h(6)/1000 + -0.0271*alpha(6)*180/pi + 0.0028*W_areades;

%Requisito
Dp_tot = Dp1c + abs(Dp2c);

if Dp_tot<Dp_max
    fprintf('Il Dp totale è pari a %.4f psf: requisito rispettato', Dp_tot)
    disp(' ')
else
    fprintf('Il Dp totale è pari a %.4f psf: requisito non rispettato', Dp_tot)
    disp(' ')
end


%% Vista in pianta aggiornata

% Fuselage configuration - Top view

nose_tan = (0.5 * F_width) / (0.15 * F_lenght);
tailcone_tan = (0.5 * F_width) / (0.20 * F_lenght);

xf = linspace(0, F_lenght, 100); % Equivalente di np.linspace in Python
y1f = zeros(1, length(xf));
y2f = zeros(1, length(xf));

for i = 1:length(xf)
    if xf(i) <= 0.15 * F_lenght
        y1f(i) = xf(i) * nose_tan;
    elseif xf(i) > 0.15 * F_lenght && xf(i) < 0.80 * F_lenght
        y1f(i) = 0.5 * F_width;
    else
        y1f(i) = (0.5 * F_width) - (xf(i) - 0.80 * F_lenght) * tailcone_tan;
    end
end

y2f = -y1f;

figure
plot(xf, y1f, 'k', xf, y2f, 'k','lineWidth',1.5)
title('Vista in pianta - Fusoliera')
ylim([-20, 20])
grid on

% Wing configuration - Top view

wing_tan_upper = (0.5 * W_span) / (W_rootC - W_tipC);
wing_tan_lower = -(0.5 * W_span) / (W_rootC - W_tipC);

xw = linspace(0, W_rootC - W_tipC, 100);
y1w = xw * wing_tan_upper;
y2w = xw * wing_tan_lower;

figure
plot(xw, y1w, 'k', xw, y2w, 'k','lineWidth',1.5)
hold on
line([W_rootC - W_tipC, W_rootC], [(W_rootC - W_tipC) * wing_tan_upper, (W_rootC - W_tipC) * wing_tan_upper], 'Color', 'k','lineWidth',1.5)
line([W_rootC - W_tipC, W_rootC], [(W_rootC - W_tipC) * wing_tan_lower, (W_rootC - W_tipC) * wing_tan_lower], 'Color', 'k','lineWidth',1.5)
line([W_rootC, W_rootC], [(W_rootC - W_tipC) * wing_tan_lower, (W_rootC - W_tipC) * wing_tan_upper], 'Color', 'k','lineWidth',1.5)
title('Vista in pianta - Ala')
grid on

% Tail configuration - Top view

x1t_root = T_rootC * [0, 0.0125, 0.025, 0.05, 0.075, 0.1, 0.15, 0.2, 0.25, 0.3, 0.4, 0.5, 0.6, ...
                           0.7, 0.8, 0.9, 0.95, 1];
x2t_root = x1t_root;
y1t_root = T_rootC * [0, 0.00947, 0.01307, 0.01777, 0.021, 0.02341, 0.02673, 0.02869, 0.02971, 0.03001, 0.02902, 0.02647, 0.02282, ...
                           0.01832, 0.01312, 0.00724, 0.00403, 0];
y2t_root = -y1t_root;

x1t_tip = T_tipC * [0, 0.0125, 0.025, 0.05, 0.075, 0.1, 0.15, 0.2, 0.25, 0.3, 0.4, 0.5, 0.6, ...
                         0.7, 0.8, 0.9, 0.95, 1];
x2t_tip = x1t_tip;
y1t_tip = T_tipC * [0, 0.00947, 0.01307, 0.01777, 0.021, 0.02341, 0.02673, 0.02869, 0.02971, 0.03001, 0.02902, 0.02647, 0.02282, ...
                         0.01832, 0.01312, 0.00724, 0.00403, 0];
y2t_tip = -y1t_tip;

figure
plot(x1t_root, y1t_root, 'k', x2t_root, y2t_root, 'k','lineWidth',1.5)
hold on
plot(x1t_tip + (T_rootC - T_tipC), y1t_tip, 'k', x2t_tip + (T_rootC - T_tipC), y2t_tip, 'k','lineWidth',1.5)
title('Vista in pianta - Coda')
ylim([-5, 5])
grid on

% Aircraft configuration - Top view

mach_angle = asin(1 / Machp); % Mach cone angle [rad]
wing_start_length = 0.25 * F_lenght;
tail_start_length = 0.70 * F_lenght;

xm = linspace(0, F_lenght, 100);
y1m = xm * tan(mach_angle);
y2m = xm * tan(-mach_angle);

figure
plot(xm, y1m, 'r', xm, y2m, 'r','lineWidth',1.5)
hold on
plot(xf, y1f, 'k', xf, y2f, 'k','lineWidth',1.5)
plot(xw + wing_start_length, y1w, 'k', xw + wing_start_length, y2w, 'k','lineWidth',1.5)
line([W_rootC - W_tipC + wing_start_length, W_rootC + wing_start_length], ...
     [(W_rootC - W_tipC) * wing_tan_upper, (W_rootC - W_tipC) * wing_tan_upper], 'Color', 'k','lineWidth',1.5)
line([W_rootC - W_tipC + wing_start_length, W_rootC + wing_start_length], ...
     [(W_rootC - W_tipC) * wing_tan_lower, (W_rootC - W_tipC) * wing_tan_lower], 'Color', 'k','lineWidth',1.5)
line([W_rootC + wing_start_length, W_rootC + wing_start_length], ...
     [(W_rootC - W_tipC) * wing_tan_lower, (W_rootC - W_tipC) * wing_tan_upper], 'Color', 'k','lineWidth',1.5)
plot(x1t_root + tail_start_length, y1t_root, 'k', x2t_root + tail_start_length, y2t_root, 'k','lineWidth',1.5)
plot(x1t_tip + (T_rootC - T_tipC) + tail_start_length, y1t_tip, 'k', x2t_tip + (T_rootC - T_tipC) + tail_start_length, y2t_tip, 'k','lineWidth',1.5)
title('Vista in pianta')
ylim([-20, 20])
grid on

% Fuselage configuration - Side view
nose_tan_upper = (0.6 * F_height) / (0.15 * F_lenght);
nose_tan_lower = -(0.4 * F_height) / (0.15 * F_lenght);
tailcone_tan_lower = F_height / (0.20 * F_lenght);

y1f = zeros(1, length(xf));
y2f = zeros(1, length(xf));

for i = 1:length(xf)
    if xf(i) <= 0.15 * F_lenght
        y1f(i) = xf(i) * nose_tan_upper;
        y2f(i) = xf(i) * nose_tan_lower;
    elseif xf(i) > 0.15 * F_lenght && xf(i) < 0.80 * F_lenght
        y1f(i) = 0.6 * F_height;
        y2f(i) = -0.4 * F_height;
    else
        y1f(i) = 0.6 * F_height;
        y2f(i) = -0.4 * F_height + (xf(i) - 0.80 * F_lenght) * tailcone_tan_lower;
    end
end

figure
plot(xf, y1f, 'k', xf, y2f, 'k','lineWidth',1.5)
title('Vista laterale - Fusoliera')
ylim([-20, 20])
grid on

% Wing configuration - Side view (Airfoil selection: NACA 65206)
x1w_root = W_rootC * [0, 0.0046, 0.00706, 0.012, 0.02444, 0.04939, 0.07437, 0.09936, 0.14939, 0.19945, 0.24953, 0.29962, 0.34971, ...
                           0.39981, 0.4499, 0.5, 0.55009, 0.60016, 0.65022, 0.70026, 0.75028, 0.80027, 0.85024, 0.90018, 0.95009, 1];
x2w_root = W_rootC * [0, 0.0054, 0.00794, 0.013, 0.02556, 0.05061, 0.07563, 0.10064, 0.15061, 0.20055, 0.25047, 0.30038, 0.35029, ...
                           0.40019, 0.4501, 0.5, 0.54991, 0.59984, 0.64978, 0.69974, 0.74972, 0.79973, 0.84976, 0.89982, 0.94991, 1];
y1w_root = W_rootC * [0, 0.00524, 0.00642, 0.00822, 0.0114, 0.01625, 0.02012, 0.0234, 0.02869, 0.03277, 0.03592, 0.03824, 0.03982, ...
                           0.04069, 0.04078, 0.04003, 0.03836, 0.03589, 0.03276, 0.02907, 0.02489, 0.02029, 0.01538, 0.01027, 0.00511, 0];
y2w_root = W_rootC * [0, -0.00424, -0.00502, -0.00608, -0.00768, -0.00993, -0.01164, -0.01306, -0.01523, -0.01685, -0.01802, -0.0188, -0.01922, ...
                           -0.01927, -0.01888, -0.01797, -0.01646, -0.01447, -0.01216, -0.00963, -0.00699, -0.00437, -0.00192, 0.00007, 0.00121, 0];
x1w_tip = W_tipC * [0, 0.0046, 0.00706, 0.012, 0.02444, 0.04939, 0.07437, 0.09936, 0.14939, 0.19945, 0.24953, 0.29962, 0.34971, ...
                         0.39981, 0.4499, 0.5, 0.55009, 0.60016, 0.65022, 0.70026, 0.75028, 0.80027, 0.85024, 0.90018, 0.95009, 1];
x2w_tip = W_tipC * [0, 0.0054, 0.00794, 0.013, 0.02556, 0.05061, 0.07563, 0.10064, 0.15061, 0.20055, 0.25047, 0.30038, 0.35029, ...
                         0.40019, 0.4501, 0.5, 0.54991, 0.59984, 0.64978, 0.69974, 0.74972, 0.79973, 0.84976, 0.89982, 0.94991, 1];
y1w_tip = W_tipC * [0, 0.00524, 0.00642, 0.00822, 0.0114, 0.01625, 0.02012, 0.0234, 0.02869, 0.03277, 0.03592, 0.03824, 0.03982, ...
                         0.04069, 0.04078, 0.04003, 0.03836, 0.03589, 0.03276, 0.02907, 0.02489, 0.02029, 0.01538, 0.01027, 0.00511, 0];
y2w_tip = W_tipC * [0, -0.00424, -0.00502, -0.00608, -0.00768, -0.00993, -0.01164, -0.01306, -0.01523, -0.01685, -0.01802, -0.0188, -0.01922, ...
                         -0.01927, -0.01888, -0.01797, -0.01646, -0.01447, -0.01216, -0.00963, -0.00699, -0.00437, -0.00192, 0.00007, 0.00121, 0];

figure
plot(x1w_root, y1w_root, 'k', x2w_root, y2w_root, 'k','lineWidth',1.5)
hold on
plot(x1w_tip + (W_rootC - W_tipC), y1w_tip, 'k', x2w_tip + (W_rootC - W_tipC), y2w_tip, 'k','lineWidth',1.5)
title('Vista laterale - Ala')
ylim([-10, 10])
grid on

% Tail configuration - Side view
tail_tan_upper = (T_span) / (T_rootC - T_tipC);

xt = linspace(0, T_rootC - T_tipC, 100);
y1t = xt * tail_tan_upper;

figure
plot(xt, y1t, 'k','lineWidth',1.5)
hold on
line([0, T_rootC], [0, 0],'Color', 'k','lineWidth',1.5)
line([T_rootC - T_tipC, T_rootC], [(T_rootC -T_tipC) * tail_tan_upper, (T_rootC - T_tipC) * tail_tan_upper], 'Color', 'k','lineWidth',1.5)
line([T_rootC, T_rootC], [0, y1t(end)], 'Color', 'k','lineWidth',1.5)
title('Vista laterale - Coda')
grid on

% Aircraft configuration - Side view
wing_start_height = -0.15 * F_height;
tail_start_height = 0.6 * F_height;

figure
plot(xm, y1m, 'r', xm, y2m, 'r','lineWidth',1.5)
hold on
plot(xf, y1f, 'k', xf, y2f, 'k','lineWidth',1.5)
plot(x1w_root + wing_start_length, y1w_root + wing_start_height, 'k', x2w_root + wing_start_length, y2w_root + wing_start_height, 'k','lineWidth',1.5)
plot(x1w_tip + (W_rootC - W_tipC) + wing_start_length, y1w_tip + wing_start_height, 'k', x2w_tip + (W_rootC - W_tipC) + wing_start_length, y2w_tip + wing_start_height, 'k','lineWidth',1.5)
plot(xt + tail_start_length, y1t + tail_start_height, 'k','lineWidth',1.5)
line([tail_start_length, T_rootC + tail_start_length], [tail_start_height, tail_start_height], 'Color', 'k','lineWidth',1.5)
line([T_rootC - T_tipC + tail_start_length, T_rootC + tail_start_length], ...
     [(T_rootC - T_tipC) * tail_tan_upper + tail_start_height, (T_rootC - T_tipC) * tail_tan_upper + tail_start_height], 'Color', 'k','lineWidth',1.5)
line([T_rootC + tail_start_length, T_rootC + tail_start_length], ...
    [tail_start_height, (T_rootC - T_tipC) * tail_tan_upper + tail_start_height], 'Color', 'k','lineWidth',1.5)
title('Vista laterale')
ylim([-20, 20])
grid on